/*
 * Copyright (C) 2023 Linux Studio Plugins Project <https://lsp-plug.in/>
 *           (C) 2023 Vladimir Sadovnikov <sadko4u@gmail.com>
 *
 * This file is part of lsp-dsp-lib
 * Created on: 19 апр. 2023 г.
 *
 * lsp-dsp-lib is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * any later version.
 *
 * lsp-dsp-lib is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with lsp-dsp-lib. If not, see <https://www.gnu.org/licenses/>.
 */

#include <lsp-plug.in/common/finally.h>
#include <lsp-plug.in/dsp/dsp.h>
#include <lsp-plug.in/test-fw/ByteBuffer.h>
#include <lsp-plug.in/test-fw/utest.h>

#include <private/utest/dsp/bitmap/util.h>

namespace lsp
{
    namespace generic
    {
        void bitmap_put_b1b8(dsp::bitmap_t *dst, const dsp::bitmap_t *src, ssize_t x, ssize_t y);
        void bitmap_add_b1b8(dsp::bitmap_t *dst, const dsp::bitmap_t *src, ssize_t x, ssize_t y);
        void bitmap_sub_b1b8(dsp::bitmap_t *dst, const dsp::bitmap_t *src, ssize_t x, ssize_t y);
        void bitmap_max_b1b8(dsp::bitmap_t *dst, const dsp::bitmap_t *src, ssize_t x, ssize_t y);
        void bitmap_min_b1b8(dsp::bitmap_t *dst, const dsp::bitmap_t *src, ssize_t x, ssize_t y);
    }
} /* namespace lsp */

namespace lsp
{
    namespace test
    {
        static uint8_t bitmap_data[] =
        {
            0x00, 0x00, 0x00, 0x00, 0x00,
            0x00, 0x00, 0x00, 0x00, 0x00,
            0x00, 0x00, 0x00, 0x00, 0x00,
            0x00, 0x00, 0x00, 0x00, 0x00,
            0x00, 0x00, 0x00, 0x00, 0x00,
            0x00, 0x00, 0x00, 0x00, 0x00,
            0x00, 0x00, 0x00, 0x00, 0x00,
            0x00, 0x00, 0x00, 0x00, 0x00,
            0x00, 0x00, 0x00, 0x00, 0x00,
            0x30, 0x00, 0xfc, 0x3f, 0xc0,
            0x30, 0x03, 0xfe, 0x3f, 0xe0,
            0x30, 0x03, 0x02, 0x30, 0x70,
            0x30, 0x07, 0x00, 0x30, 0x30,
            0x30, 0x07, 0x00, 0x30, 0x38,
            0x30, 0x07, 0x00, 0x30, 0x38,
            0x30, 0x03, 0x80, 0x30, 0x30,
            0x30, 0x01, 0xe0, 0x30, 0x70,
            0x30, 0x00, 0xf8, 0x3f, 0xe0,
            0x30, 0x00, 0x3c, 0x3f, 0x80,
            0x30, 0x00, 0x0e, 0x30, 0x00,
            0x30, 0x00, 0x06, 0x30, 0x00,
            0x30, 0x00, 0x06, 0x30, 0x00,
            0x30, 0x00, 0x06, 0x30, 0x00,
            0x30, 0x04, 0x0e, 0x30, 0x00,
            0x3f, 0xe7, 0xfc, 0x30, 0x00,
            0x3f, 0xe3, 0xf0, 0x30, 0x00,
            0x00, 0x00, 0x00, 0x00, 0x00,
            0x00, 0x00, 0x00, 0x00, 0x00,
            0x00, 0x00, 0x00, 0x00, 0x00,
            0x00, 0x00, 0x00, 0x00, 0x00,
            0x00, 0x00, 0x00, 0x00, 0x00,
            0x00, 0x00, 0x00, 0x00, 0x00,
            0x00, 0x00, 0x00, 0x00, 0x00,
        };

        static const lsp::dsp::bitmap_t bitmap =
        {
            38, // width
            33, // height
            5, // stride
            bitmap_data // data
        };

        typedef void (* bitmap_func_b1b8_t)(lsp::dsp::bitmap_t *dst, const lsp::dsp::bitmap_t *src, ssize_t x, ssize_t y);

        static const bitmap_pattern_t put_patterns[] =
        {
            {
                0, 0,
                0x88,
                "0000000000000000000000000000000000000000000000000000000000000000000000000000\n"
                "0000000000000000000000000000000000000000000000000000000000000000000000000000\n"
                "0000000000000000000000000000000000000000000000000000000000000000000000000000\n"
                "0000000000000000000000000000000000000000000000000000000000000000000000000000\n"
                "0000000000000000000000000000000000000000000000000000000000000000000000000000\n"
                "0000000000000000000000000000000000000000000000000000000000000000000000000000\n"
                "0000000000000000000000000000000000000000000000000000000000000000000000000000\n"
                "0000000000000000000000000000000000000000000000000000000000000000000000000000\n"
                "0000000000000000000000000000000000000000000000000000000000000000000000000000\n"
                "0000ffff000000000000000000000000ffffffffffff00000000ffffffffffffffff00000000\n"
                "0000ffff00000000000000000000ffffffffffffffffff000000ffffffffffffffffff000000\n"
                "0000ffff00000000000000000000ffff000000000000ff000000ffff0000000000ffffff0000\n"
                "0000ffff000000000000000000ffffff00000000000000000000ffff000000000000ffff0000\n"
                "0000ffff000000000000000000ffffff00000000000000000000ffff000000000000ffffff00\n"
                "0000ffff000000000000000000ffffff00000000000000000000ffff000000000000ffffff00\n"
                "0000ffff00000000000000000000ffffff000000000000000000ffff000000000000ffff0000\n"
                "0000ffff0000000000000000000000ffffffff00000000000000ffff0000000000ffffff0000\n"
                "0000ffff000000000000000000000000ffffffffff0000000000ffffffffffffffffff000000\n"
                "0000ffff0000000000000000000000000000ffffffff00000000ffffffffffffff0000000000\n"
                "0000ffff00000000000000000000000000000000ffffff000000ffff00000000000000000000\n"
                "0000ffff0000000000000000000000000000000000ffff000000ffff00000000000000000000\n"
                "0000ffff0000000000000000000000000000000000ffff000000ffff00000000000000000000\n"
                "0000ffff0000000000000000000000000000000000ffff000000ffff00000000000000000000\n"
                "0000ffff000000000000000000ff000000000000ffffff000000ffff00000000000000000000\n"
                "0000ffffffffffffffffff0000ffffffffffffffffff00000000ffff00000000000000000000\n"
                "0000ffffffffffffffffff000000ffffffffffff000000000000ffff00000000000000000000\n"
                "0000000000000000000000000000000000000000000000000000000000000000000000000000\n"
                "0000000000000000000000000000000000000000000000000000000000000000000000000000\n"
                "0000000000000000000000000000000000000000000000000000000000000000000000000000\n"
                "0000000000000000000000000000000000000000000000000000000000000000000000000000\n"
                "0000000000000000000000000000000000000000000000000000000000000000000000000000\n"
                "0000000000000000000000000000000000000000000000000000000000000000000000000000\n"
                "0000000000000000000000000000000000000000000000000000000000000000000000000000\n"
            },
            {
                -8, -8,
                0x88,
                "0000000000000000000000000000000000000000000000000000000000008888888888888888\n"
                "0000000000000000ffffffffffff00000000ffffffffffffffff000000008888888888888888\n"
                "000000000000ffffffffffffffffff000000ffffffffffffffffff0000008888888888888888\n"
                "000000000000ffff000000000000ff000000ffff0000000000ffffff00008888888888888888\n"
                "0000000000ffffff00000000000000000000ffff000000000000ffff00008888888888888888\n"
                "0000000000ffffff00000000000000000000ffff000000000000ffffff008888888888888888\n"
                "0000000000ffffff00000000000000000000ffff000000000000ffffff008888888888888888\n"
                "000000000000ffffff000000000000000000ffff000000000000ffff00008888888888888888\n"
                "00000000000000ffffffff00000000000000ffff0000000000ffffff00008888888888888888\n"
                "0000000000000000ffffffffff0000000000ffffffffffffffffff0000008888888888888888\n"
                "00000000000000000000ffffffff00000000ffffffffffffff00000000008888888888888888\n"
                "000000000000000000000000ffffff000000ffff000000000000000000008888888888888888\n"
                "00000000000000000000000000ffff000000ffff000000000000000000008888888888888888\n"
                "00000000000000000000000000ffff000000ffff000000000000000000008888888888888888\n"
                "00000000000000000000000000ffff000000ffff000000000000000000008888888888888888\n"
                "0000000000ff000000000000ffffff000000ffff000000000000000000008888888888888888\n"
                "ffffff0000ffffffffffffffffff00000000ffff000000000000000000008888888888888888\n"
                "ffffff000000ffffffffffff000000000000ffff000000000000000000008888888888888888\n"
                "0000000000000000000000000000000000000000000000000000000000008888888888888888\n"
                "0000000000000000000000000000000000000000000000000000000000008888888888888888\n"
                "0000000000000000000000000000000000000000000000000000000000008888888888888888\n"
                "0000000000000000000000000000000000000000000000000000000000008888888888888888\n"
                "0000000000000000000000000000000000000000000000000000000000008888888888888888\n"
                "0000000000000000000000000000000000000000000000000000000000008888888888888888\n"
                "0000000000000000000000000000000000000000000000000000000000008888888888888888\n"
                "8888888888888888888888888888888888888888888888888888888888888888888888888888\n"
                "8888888888888888888888888888888888888888888888888888888888888888888888888888\n"
                "8888888888888888888888888888888888888888888888888888888888888888888888888888\n"
                "8888888888888888888888888888888888888888888888888888888888888888888888888888\n"
                "8888888888888888888888888888888888888888888888888888888888888888888888888888\n"
                "8888888888888888888888888888888888888888888888888888888888888888888888888888\n"
                "8888888888888888888888888888888888888888888888888888888888888888888888888888\n"
                "8888888888888888888888888888888888888888888888888888888888888888888888888888\n"
            },
            {
                8, -8,
                0x88,
                "8888888888888888000000000000000000000000000000000000000000000000000000000000\n"
                "88888888888888880000ffff000000000000000000000000ffffffffffff00000000ffffffff\n"
                "88888888888888880000ffff00000000000000000000ffffffffffffffffff000000ffffffff\n"
                "88888888888888880000ffff00000000000000000000ffff000000000000ff000000ffff0000\n"
                "88888888888888880000ffff000000000000000000ffffff00000000000000000000ffff0000\n"
                "88888888888888880000ffff000000000000000000ffffff00000000000000000000ffff0000\n"
                "88888888888888880000ffff000000000000000000ffffff00000000000000000000ffff0000\n"
                "88888888888888880000ffff00000000000000000000ffffff000000000000000000ffff0000\n"
                "88888888888888880000ffff0000000000000000000000ffffffff00000000000000ffff0000\n"
                "88888888888888880000ffff000000000000000000000000ffffffffff0000000000ffffffff\n"
                "88888888888888880000ffff0000000000000000000000000000ffffffff00000000ffffffff\n"
                "88888888888888880000ffff00000000000000000000000000000000ffffff000000ffff0000\n"
                "88888888888888880000ffff0000000000000000000000000000000000ffff000000ffff0000\n"
                "88888888888888880000ffff0000000000000000000000000000000000ffff000000ffff0000\n"
                "88888888888888880000ffff0000000000000000000000000000000000ffff000000ffff0000\n"
                "88888888888888880000ffff000000000000000000ff000000000000ffffff000000ffff0000\n"
                "88888888888888880000ffffffffffffffffff0000ffffffffffffffffff00000000ffff0000\n"
                "88888888888888880000ffffffffffffffffff000000ffffffffffff000000000000ffff0000\n"
                "8888888888888888000000000000000000000000000000000000000000000000000000000000\n"
                "8888888888888888000000000000000000000000000000000000000000000000000000000000\n"
                "8888888888888888000000000000000000000000000000000000000000000000000000000000\n"
                "8888888888888888000000000000000000000000000000000000000000000000000000000000\n"
                "8888888888888888000000000000000000000000000000000000000000000000000000000000\n"
                "8888888888888888000000000000000000000000000000000000000000000000000000000000\n"
                "8888888888888888000000000000000000000000000000000000000000000000000000000000\n"
                "8888888888888888888888888888888888888888888888888888888888888888888888888888\n"
                "8888888888888888888888888888888888888888888888888888888888888888888888888888\n"
                "8888888888888888888888888888888888888888888888888888888888888888888888888888\n"
                "8888888888888888888888888888888888888888888888888888888888888888888888888888\n"
                "8888888888888888888888888888888888888888888888888888888888888888888888888888\n"
                "8888888888888888888888888888888888888888888888888888888888888888888888888888\n"
                "8888888888888888888888888888888888888888888888888888888888888888888888888888\n"
                "8888888888888888888888888888888888888888888888888888888888888888888888888888\n"
            },
            {
                8, 8,
                0x88,
                "8888888888888888888888888888888888888888888888888888888888888888888888888888\n"
                "8888888888888888888888888888888888888888888888888888888888888888888888888888\n"
                "8888888888888888888888888888888888888888888888888888888888888888888888888888\n"
                "8888888888888888888888888888888888888888888888888888888888888888888888888888\n"
                "8888888888888888888888888888888888888888888888888888888888888888888888888888\n"
                "8888888888888888888888888888888888888888888888888888888888888888888888888888\n"
                "8888888888888888888888888888888888888888888888888888888888888888888888888888\n"
                "8888888888888888888888888888888888888888888888888888888888888888888888888888\n"
                "8888888888888888000000000000000000000000000000000000000000000000000000000000\n"
                "8888888888888888000000000000000000000000000000000000000000000000000000000000\n"
                "8888888888888888000000000000000000000000000000000000000000000000000000000000\n"
                "8888888888888888000000000000000000000000000000000000000000000000000000000000\n"
                "8888888888888888000000000000000000000000000000000000000000000000000000000000\n"
                "8888888888888888000000000000000000000000000000000000000000000000000000000000\n"
                "8888888888888888000000000000000000000000000000000000000000000000000000000000\n"
                "8888888888888888000000000000000000000000000000000000000000000000000000000000\n"
                "8888888888888888000000000000000000000000000000000000000000000000000000000000\n"
                "88888888888888880000ffff000000000000000000000000ffffffffffff00000000ffffffff\n"
                "88888888888888880000ffff00000000000000000000ffffffffffffffffff000000ffffffff\n"
                "88888888888888880000ffff00000000000000000000ffff000000000000ff000000ffff0000\n"
                "88888888888888880000ffff000000000000000000ffffff00000000000000000000ffff0000\n"
                "88888888888888880000ffff000000000000000000ffffff00000000000000000000ffff0000\n"
                "88888888888888880000ffff000000000000000000ffffff00000000000000000000ffff0000\n"
                "88888888888888880000ffff00000000000000000000ffffff000000000000000000ffff0000\n"
                "88888888888888880000ffff0000000000000000000000ffffffff00000000000000ffff0000\n"
                "88888888888888880000ffff000000000000000000000000ffffffffff0000000000ffffffff\n"
                "88888888888888880000ffff0000000000000000000000000000ffffffff00000000ffffffff\n"
                "88888888888888880000ffff00000000000000000000000000000000ffffff000000ffff0000\n"
                "88888888888888880000ffff0000000000000000000000000000000000ffff000000ffff0000\n"
                "88888888888888880000ffff0000000000000000000000000000000000ffff000000ffff0000\n"
                "88888888888888880000ffff0000000000000000000000000000000000ffff000000ffff0000\n"
                "88888888888888880000ffff000000000000000000ff000000000000ffffff000000ffff0000\n"
                "88888888888888880000ffffffffffffffffff0000ffffffffffffffffff00000000ffff0000\n"
            },
            {
                -8, 8,
                0x88,
                "8888888888888888888888888888888888888888888888888888888888888888888888888888\n"
                "8888888888888888888888888888888888888888888888888888888888888888888888888888\n"
                "8888888888888888888888888888888888888888888888888888888888888888888888888888\n"
                "8888888888888888888888888888888888888888888888888888888888888888888888888888\n"
                "8888888888888888888888888888888888888888888888888888888888888888888888888888\n"
                "8888888888888888888888888888888888888888888888888888888888888888888888888888\n"
                "8888888888888888888888888888888888888888888888888888888888888888888888888888\n"
                "8888888888888888888888888888888888888888888888888888888888888888888888888888\n"
                "0000000000000000000000000000000000000000000000000000000000008888888888888888\n"
                "0000000000000000000000000000000000000000000000000000000000008888888888888888\n"
                "0000000000000000000000000000000000000000000000000000000000008888888888888888\n"
                "0000000000000000000000000000000000000000000000000000000000008888888888888888\n"
                "0000000000000000000000000000000000000000000000000000000000008888888888888888\n"
                "0000000000000000000000000000000000000000000000000000000000008888888888888888\n"
                "0000000000000000000000000000000000000000000000000000000000008888888888888888\n"
                "0000000000000000000000000000000000000000000000000000000000008888888888888888\n"
                "0000000000000000000000000000000000000000000000000000000000008888888888888888\n"
                "0000000000000000ffffffffffff00000000ffffffffffffffff000000008888888888888888\n"
                "000000000000ffffffffffffffffff000000ffffffffffffffffff0000008888888888888888\n"
                "000000000000ffff000000000000ff000000ffff0000000000ffffff00008888888888888888\n"
                "0000000000ffffff00000000000000000000ffff000000000000ffff00008888888888888888\n"
                "0000000000ffffff00000000000000000000ffff000000000000ffffff008888888888888888\n"
                "0000000000ffffff00000000000000000000ffff000000000000ffffff008888888888888888\n"
                "000000000000ffffff000000000000000000ffff000000000000ffff00008888888888888888\n"
                "00000000000000ffffffff00000000000000ffff0000000000ffffff00008888888888888888\n"
                "0000000000000000ffffffffff0000000000ffffffffffffffffff0000008888888888888888\n"
                "00000000000000000000ffffffff00000000ffffffffffffff00000000008888888888888888\n"
                "000000000000000000000000ffffff000000ffff000000000000000000008888888888888888\n"
                "00000000000000000000000000ffff000000ffff000000000000000000008888888888888888\n"
                "00000000000000000000000000ffff000000ffff000000000000000000008888888888888888\n"
                "00000000000000000000000000ffff000000ffff000000000000000000008888888888888888\n"
                "0000000000ff000000000000ffffff000000ffff000000000000000000008888888888888888\n"
                "ffffff0000ffffffffffffffffff00000000ffff000000000000000000008888888888888888\n"
            },
            {
                0, 0,
                0,
                NULL
            }
        };

        static const bitmap_pattern_t add_patterns[] =
        {
            {
                0, 0,
                0x88,
                "8888888888888888888888888888888888888888888888888888888888888888888888888888\n"
                "8888888888888888888888888888888888888888888888888888888888888888888888888888\n"
                "8888888888888888888888888888888888888888888888888888888888888888888888888888\n"
                "8888888888888888888888888888888888888888888888888888888888888888888888888888\n"
                "8888888888888888888888888888888888888888888888888888888888888888888888888888\n"
                "8888888888888888888888888888888888888888888888888888888888888888888888888888\n"
                "8888888888888888888888888888888888888888888888888888888888888888888888888888\n"
                "8888888888888888888888888888888888888888888888888888888888888888888888888888\n"
                "8888888888888888888888888888888888888888888888888888888888888888888888888888\n"
                "8888ffff888888888888888888888888ffffffffffff88888888ffffffffffffffff88888888\n"
                "8888ffff88888888888888888888ffffffffffffffffff888888ffffffffffffffffff888888\n"
                "8888ffff88888888888888888888ffff888888888888ff888888ffff8888888888ffffff8888\n"
                "8888ffff888888888888888888ffffff88888888888888888888ffff888888888888ffff8888\n"
                "8888ffff888888888888888888ffffff88888888888888888888ffff888888888888ffffff88\n"
                "8888ffff888888888888888888ffffff88888888888888888888ffff888888888888ffffff88\n"
                "8888ffff88888888888888888888ffffff888888888888888888ffff888888888888ffff8888\n"
                "8888ffff8888888888888888888888ffffffff88888888888888ffff8888888888ffffff8888\n"
                "8888ffff888888888888888888888888ffffffffff8888888888ffffffffffffffffff888888\n"
                "8888ffff8888888888888888888888888888ffffffff88888888ffffffffffffff8888888888\n"
                "8888ffff88888888888888888888888888888888ffffff888888ffff88888888888888888888\n"
                "8888ffff8888888888888888888888888888888888ffff888888ffff88888888888888888888\n"
                "8888ffff8888888888888888888888888888888888ffff888888ffff88888888888888888888\n"
                "8888ffff8888888888888888888888888888888888ffff888888ffff88888888888888888888\n"
                "8888ffff888888888888888888ff888888888888ffffff888888ffff88888888888888888888\n"
                "8888ffffffffffffffffff8888ffffffffffffffffff88888888ffff88888888888888888888\n"
                "8888ffffffffffffffffff888888ffffffffffff888888888888ffff88888888888888888888\n"
                "8888888888888888888888888888888888888888888888888888888888888888888888888888\n"
                "8888888888888888888888888888888888888888888888888888888888888888888888888888\n"
                "8888888888888888888888888888888888888888888888888888888888888888888888888888\n"
                "8888888888888888888888888888888888888888888888888888888888888888888888888888\n"
                "8888888888888888888888888888888888888888888888888888888888888888888888888888\n"
                "8888888888888888888888888888888888888888888888888888888888888888888888888888\n"
                "8888888888888888888888888888888888888888888888888888888888888888888888888888\n"
            },
            {
                -8, -8,
                0x88,
                "8888888888888888888888888888888888888888888888888888888888888888888888888888\n"
                "8888888888888888ffffffffffff88888888ffffffffffffffff888888888888888888888888\n"
                "888888888888ffffffffffffffffff888888ffffffffffffffffff8888888888888888888888\n"
                "888888888888ffff888888888888ff888888ffff8888888888ffffff88888888888888888888\n"
                "8888888888ffffff88888888888888888888ffff888888888888ffff88888888888888888888\n"
                "8888888888ffffff88888888888888888888ffff888888888888ffffff888888888888888888\n"
                "8888888888ffffff88888888888888888888ffff888888888888ffffff888888888888888888\n"
                "888888888888ffffff888888888888888888ffff888888888888ffff88888888888888888888\n"
                "88888888888888ffffffff88888888888888ffff8888888888ffffff88888888888888888888\n"
                "8888888888888888ffffffffff8888888888ffffffffffffffffff8888888888888888888888\n"
                "88888888888888888888ffffffff88888888ffffffffffffff88888888888888888888888888\n"
                "888888888888888888888888ffffff888888ffff888888888888888888888888888888888888\n"
                "88888888888888888888888888ffff888888ffff888888888888888888888888888888888888\n"
                "88888888888888888888888888ffff888888ffff888888888888888888888888888888888888\n"
                "88888888888888888888888888ffff888888ffff888888888888888888888888888888888888\n"
                "8888888888ff888888888888ffffff888888ffff888888888888888888888888888888888888\n"
                "ffffff8888ffffffffffffffffff88888888ffff888888888888888888888888888888888888\n"
                "ffffff888888ffffffffffff888888888888ffff888888888888888888888888888888888888\n"
                "8888888888888888888888888888888888888888888888888888888888888888888888888888\n"
                "8888888888888888888888888888888888888888888888888888888888888888888888888888\n"
                "8888888888888888888888888888888888888888888888888888888888888888888888888888\n"
                "8888888888888888888888888888888888888888888888888888888888888888888888888888\n"
                "8888888888888888888888888888888888888888888888888888888888888888888888888888\n"
                "8888888888888888888888888888888888888888888888888888888888888888888888888888\n"
                "8888888888888888888888888888888888888888888888888888888888888888888888888888\n"
                "8888888888888888888888888888888888888888888888888888888888888888888888888888\n"
                "8888888888888888888888888888888888888888888888888888888888888888888888888888\n"
                "8888888888888888888888888888888888888888888888888888888888888888888888888888\n"
                "8888888888888888888888888888888888888888888888888888888888888888888888888888\n"
                "8888888888888888888888888888888888888888888888888888888888888888888888888888\n"
                "8888888888888888888888888888888888888888888888888888888888888888888888888888\n"
                "8888888888888888888888888888888888888888888888888888888888888888888888888888\n"
                "8888888888888888888888888888888888888888888888888888888888888888888888888888\n"
            },
            {
                8, -8,
                0x88,
                "8888888888888888888888888888888888888888888888888888888888888888888888888888\n"
                "88888888888888888888ffff888888888888888888888888ffffffffffff88888888ffffffff\n"
                "88888888888888888888ffff88888888888888888888ffffffffffffffffff888888ffffffff\n"
                "88888888888888888888ffff88888888888888888888ffff888888888888ff888888ffff8888\n"
                "88888888888888888888ffff888888888888888888ffffff88888888888888888888ffff8888\n"
                "88888888888888888888ffff888888888888888888ffffff88888888888888888888ffff8888\n"
                "88888888888888888888ffff888888888888888888ffffff88888888888888888888ffff8888\n"
                "88888888888888888888ffff88888888888888888888ffffff888888888888888888ffff8888\n"
                "88888888888888888888ffff8888888888888888888888ffffffff88888888888888ffff8888\n"
                "88888888888888888888ffff888888888888888888888888ffffffffff8888888888ffffffff\n"
                "88888888888888888888ffff8888888888888888888888888888ffffffff88888888ffffffff\n"
                "88888888888888888888ffff88888888888888888888888888888888ffffff888888ffff8888\n"
                "88888888888888888888ffff8888888888888888888888888888888888ffff888888ffff8888\n"
                "88888888888888888888ffff8888888888888888888888888888888888ffff888888ffff8888\n"
                "88888888888888888888ffff8888888888888888888888888888888888ffff888888ffff8888\n"
                "88888888888888888888ffff888888888888888888ff888888888888ffffff888888ffff8888\n"
                "88888888888888888888ffffffffffffffffff8888ffffffffffffffffff88888888ffff8888\n"
                "88888888888888888888ffffffffffffffffff888888ffffffffffff888888888888ffff8888\n"
                "8888888888888888888888888888888888888888888888888888888888888888888888888888\n"
                "8888888888888888888888888888888888888888888888888888888888888888888888888888\n"
                "8888888888888888888888888888888888888888888888888888888888888888888888888888\n"
                "8888888888888888888888888888888888888888888888888888888888888888888888888888\n"
                "8888888888888888888888888888888888888888888888888888888888888888888888888888\n"
                "8888888888888888888888888888888888888888888888888888888888888888888888888888\n"
                "8888888888888888888888888888888888888888888888888888888888888888888888888888\n"
                "8888888888888888888888888888888888888888888888888888888888888888888888888888\n"
                "8888888888888888888888888888888888888888888888888888888888888888888888888888\n"
                "8888888888888888888888888888888888888888888888888888888888888888888888888888\n"
                "8888888888888888888888888888888888888888888888888888888888888888888888888888\n"
                "8888888888888888888888888888888888888888888888888888888888888888888888888888\n"
                "8888888888888888888888888888888888888888888888888888888888888888888888888888\n"
                "8888888888888888888888888888888888888888888888888888888888888888888888888888\n"
                "8888888888888888888888888888888888888888888888888888888888888888888888888888\n"
            },
            {
                8, 8,
                0x88,
                "8888888888888888888888888888888888888888888888888888888888888888888888888888\n"
                "8888888888888888888888888888888888888888888888888888888888888888888888888888\n"
                "8888888888888888888888888888888888888888888888888888888888888888888888888888\n"
                "8888888888888888888888888888888888888888888888888888888888888888888888888888\n"
                "8888888888888888888888888888888888888888888888888888888888888888888888888888\n"
                "8888888888888888888888888888888888888888888888888888888888888888888888888888\n"
                "8888888888888888888888888888888888888888888888888888888888888888888888888888\n"
                "8888888888888888888888888888888888888888888888888888888888888888888888888888\n"
                "8888888888888888888888888888888888888888888888888888888888888888888888888888\n"
                "8888888888888888888888888888888888888888888888888888888888888888888888888888\n"
                "8888888888888888888888888888888888888888888888888888888888888888888888888888\n"
                "8888888888888888888888888888888888888888888888888888888888888888888888888888\n"
                "8888888888888888888888888888888888888888888888888888888888888888888888888888\n"
                "8888888888888888888888888888888888888888888888888888888888888888888888888888\n"
                "8888888888888888888888888888888888888888888888888888888888888888888888888888\n"
                "8888888888888888888888888888888888888888888888888888888888888888888888888888\n"
                "8888888888888888888888888888888888888888888888888888888888888888888888888888\n"
                "88888888888888888888ffff888888888888888888888888ffffffffffff88888888ffffffff\n"
                "88888888888888888888ffff88888888888888888888ffffffffffffffffff888888ffffffff\n"
                "88888888888888888888ffff88888888888888888888ffff888888888888ff888888ffff8888\n"
                "88888888888888888888ffff888888888888888888ffffff88888888888888888888ffff8888\n"
                "88888888888888888888ffff888888888888888888ffffff88888888888888888888ffff8888\n"
                "88888888888888888888ffff888888888888888888ffffff88888888888888888888ffff8888\n"
                "88888888888888888888ffff88888888888888888888ffffff888888888888888888ffff8888\n"
                "88888888888888888888ffff8888888888888888888888ffffffff88888888888888ffff8888\n"
                "88888888888888888888ffff888888888888888888888888ffffffffff8888888888ffffffff\n"
                "88888888888888888888ffff8888888888888888888888888888ffffffff88888888ffffffff\n"
                "88888888888888888888ffff88888888888888888888888888888888ffffff888888ffff8888\n"
                "88888888888888888888ffff8888888888888888888888888888888888ffff888888ffff8888\n"
                "88888888888888888888ffff8888888888888888888888888888888888ffff888888ffff8888\n"
                "88888888888888888888ffff8888888888888888888888888888888888ffff888888ffff8888\n"
                "88888888888888888888ffff888888888888888888ff888888888888ffffff888888ffff8888\n"
                "88888888888888888888ffffffffffffffffff8888ffffffffffffffffff88888888ffff8888\n"
            },
            {
                -8, 8,
                0x88,
                "8888888888888888888888888888888888888888888888888888888888888888888888888888\n"
                "8888888888888888888888888888888888888888888888888888888888888888888888888888\n"
                "8888888888888888888888888888888888888888888888888888888888888888888888888888\n"
                "8888888888888888888888888888888888888888888888888888888888888888888888888888\n"
                "8888888888888888888888888888888888888888888888888888888888888888888888888888\n"
                "8888888888888888888888888888888888888888888888888888888888888888888888888888\n"
                "8888888888888888888888888888888888888888888888888888888888888888888888888888\n"
                "8888888888888888888888888888888888888888888888888888888888888888888888888888\n"
                "8888888888888888888888888888888888888888888888888888888888888888888888888888\n"
                "8888888888888888888888888888888888888888888888888888888888888888888888888888\n"
                "8888888888888888888888888888888888888888888888888888888888888888888888888888\n"
                "8888888888888888888888888888888888888888888888888888888888888888888888888888\n"
                "8888888888888888888888888888888888888888888888888888888888888888888888888888\n"
                "8888888888888888888888888888888888888888888888888888888888888888888888888888\n"
                "8888888888888888888888888888888888888888888888888888888888888888888888888888\n"
                "8888888888888888888888888888888888888888888888888888888888888888888888888888\n"
                "8888888888888888888888888888888888888888888888888888888888888888888888888888\n"
                "8888888888888888ffffffffffff88888888ffffffffffffffff888888888888888888888888\n"
                "888888888888ffffffffffffffffff888888ffffffffffffffffff8888888888888888888888\n"
                "888888888888ffff888888888888ff888888ffff8888888888ffffff88888888888888888888\n"
                "8888888888ffffff88888888888888888888ffff888888888888ffff88888888888888888888\n"
                "8888888888ffffff88888888888888888888ffff888888888888ffffff888888888888888888\n"
                "8888888888ffffff88888888888888888888ffff888888888888ffffff888888888888888888\n"
                "888888888888ffffff888888888888888888ffff888888888888ffff88888888888888888888\n"
                "88888888888888ffffffff88888888888888ffff8888888888ffffff88888888888888888888\n"
                "8888888888888888ffffffffff8888888888ffffffffffffffffff8888888888888888888888\n"
                "88888888888888888888ffffffff88888888ffffffffffffff88888888888888888888888888\n"
                "888888888888888888888888ffffff888888ffff888888888888888888888888888888888888\n"
                "88888888888888888888888888ffff888888ffff888888888888888888888888888888888888\n"
                "88888888888888888888888888ffff888888ffff888888888888888888888888888888888888\n"
                "88888888888888888888888888ffff888888ffff888888888888888888888888888888888888\n"
                "8888888888ff888888888888ffffff888888ffff888888888888888888888888888888888888\n"
                "ffffff8888ffffffffffffffffff88888888ffff888888888888888888888888888888888888\n"
            },
            {
                0, 0,
                0,
                NULL
            }
        };

        static const bitmap_pattern_t sub_patterns[] =
        {
            {
                0, 0,
                0x88,
                "8888888888888888888888888888888888888888888888888888888888888888888888888888\n"
                "8888888888888888888888888888888888888888888888888888888888888888888888888888\n"
                "8888888888888888888888888888888888888888888888888888888888888888888888888888\n"
                "8888888888888888888888888888888888888888888888888888888888888888888888888888\n"
                "8888888888888888888888888888888888888888888888888888888888888888888888888888\n"
                "8888888888888888888888888888888888888888888888888888888888888888888888888888\n"
                "8888888888888888888888888888888888888888888888888888888888888888888888888888\n"
                "8888888888888888888888888888888888888888888888888888888888888888888888888888\n"
                "8888888888888888888888888888888888888888888888888888888888888888888888888888\n"
                "8888000088888888888888888888888800000000000088888888000000000000000088888888\n"
                "8888000088888888888888888888000000000000000000888888000000000000000000888888\n"
                "8888000088888888888888888888000088888888888800888888000088888888880000008888\n"
                "8888000088888888888888888800000088888888888888888888000088888888888800008888\n"
                "8888000088888888888888888800000088888888888888888888000088888888888800000088\n"
                "8888000088888888888888888800000088888888888888888888000088888888888800000088\n"
                "8888000088888888888888888888000000888888888888888888000088888888888800008888\n"
                "8888000088888888888888888888880000000088888888888888000088888888880000008888\n"
                "8888000088888888888888888888888800000000008888888888000000000000000000888888\n"
                "8888000088888888888888888888888888880000000088888888000000000000008888888888\n"
                "8888000088888888888888888888888888888888000000888888000088888888888888888888\n"
                "8888000088888888888888888888888888888888880000888888000088888888888888888888\n"
                "8888000088888888888888888888888888888888880000888888000088888888888888888888\n"
                "8888000088888888888888888888888888888888880000888888000088888888888888888888\n"
                "8888000088888888888888888800888888888888000000888888000088888888888888888888\n"
                "8888000000000000000000888800000000000000000088888888000088888888888888888888\n"
                "8888000000000000000000888888000000000000888888888888000088888888888888888888\n"
                "8888888888888888888888888888888888888888888888888888888888888888888888888888\n"
                "8888888888888888888888888888888888888888888888888888888888888888888888888888\n"
                "8888888888888888888888888888888888888888888888888888888888888888888888888888\n"
                "8888888888888888888888888888888888888888888888888888888888888888888888888888\n"
                "8888888888888888888888888888888888888888888888888888888888888888888888888888\n"
                "8888888888888888888888888888888888888888888888888888888888888888888888888888\n"
                "8888888888888888888888888888888888888888888888888888888888888888888888888888\n"
            },
            {
                -8, -8,
                0x88,
                "8888888888888888888888888888888888888888888888888888888888888888888888888888\n"
                "8888888888888888000000000000888888880000000000000000888888888888888888888888\n"
                "8888888888880000000000000000008888880000000000000000008888888888888888888888\n"
                "8888888888880000888888888888008888880000888888888800000088888888888888888888\n"
                "8888888888000000888888888888888888880000888888888888000088888888888888888888\n"
                "8888888888000000888888888888888888880000888888888888000000888888888888888888\n"
                "8888888888000000888888888888888888880000888888888888000000888888888888888888\n"
                "8888888888880000008888888888888888880000888888888888000088888888888888888888\n"
                "8888888888888800000000888888888888880000888888888800000088888888888888888888\n"
                "8888888888888888000000000088888888880000000000000000008888888888888888888888\n"
                "8888888888888888888800000000888888880000000000000088888888888888888888888888\n"
                "8888888888888888888888880000008888880000888888888888888888888888888888888888\n"
                "8888888888888888888888888800008888880000888888888888888888888888888888888888\n"
                "8888888888888888888888888800008888880000888888888888888888888888888888888888\n"
                "8888888888888888888888888800008888880000888888888888888888888888888888888888\n"
                "8888888888008888888888880000008888880000888888888888888888888888888888888888\n"
                "0000008888000000000000000000888888880000888888888888888888888888888888888888\n"
                "0000008888880000000000008888888888880000888888888888888888888888888888888888\n"
                "8888888888888888888888888888888888888888888888888888888888888888888888888888\n"
                "8888888888888888888888888888888888888888888888888888888888888888888888888888\n"
                "8888888888888888888888888888888888888888888888888888888888888888888888888888\n"
                "8888888888888888888888888888888888888888888888888888888888888888888888888888\n"
                "8888888888888888888888888888888888888888888888888888888888888888888888888888\n"
                "8888888888888888888888888888888888888888888888888888888888888888888888888888\n"
                "8888888888888888888888888888888888888888888888888888888888888888888888888888\n"
                "8888888888888888888888888888888888888888888888888888888888888888888888888888\n"
                "8888888888888888888888888888888888888888888888888888888888888888888888888888\n"
                "8888888888888888888888888888888888888888888888888888888888888888888888888888\n"
                "8888888888888888888888888888888888888888888888888888888888888888888888888888\n"
                "8888888888888888888888888888888888888888888888888888888888888888888888888888\n"
                "8888888888888888888888888888888888888888888888888888888888888888888888888888\n"
                "8888888888888888888888888888888888888888888888888888888888888888888888888888\n"
                "8888888888888888888888888888888888888888888888888888888888888888888888888888\n"
            },
            {
                8, -8,
                0x88,
                "8888888888888888888888888888888888888888888888888888888888888888888888888888\n"
                "8888888888888888888800008888888888888888888888880000000000008888888800000000\n"
                "8888888888888888888800008888888888888888888800000000000000000088888800000000\n"
                "8888888888888888888800008888888888888888888800008888888888880088888800008888\n"
                "8888888888888888888800008888888888888888880000008888888888888888888800008888\n"
                "8888888888888888888800008888888888888888880000008888888888888888888800008888\n"
                "8888888888888888888800008888888888888888880000008888888888888888888800008888\n"
                "8888888888888888888800008888888888888888888800000088888888888888888800008888\n"
                "8888888888888888888800008888888888888888888888000000008888888888888800008888\n"
                "8888888888888888888800008888888888888888888888880000000000888888888800000000\n"
                "8888888888888888888800008888888888888888888888888888000000008888888800000000\n"
                "8888888888888888888800008888888888888888888888888888888800000088888800008888\n"
                "8888888888888888888800008888888888888888888888888888888888000088888800008888\n"
                "8888888888888888888800008888888888888888888888888888888888000088888800008888\n"
                "8888888888888888888800008888888888888888888888888888888888000088888800008888\n"
                "8888888888888888888800008888888888888888880088888888888800000088888800008888\n"
                "8888888888888888888800000000000000000088880000000000000000008888888800008888\n"
                "8888888888888888888800000000000000000088888800000000000088888888888800008888\n"
                "8888888888888888888888888888888888888888888888888888888888888888888888888888\n"
                "8888888888888888888888888888888888888888888888888888888888888888888888888888\n"
                "8888888888888888888888888888888888888888888888888888888888888888888888888888\n"
                "8888888888888888888888888888888888888888888888888888888888888888888888888888\n"
                "8888888888888888888888888888888888888888888888888888888888888888888888888888\n"
                "8888888888888888888888888888888888888888888888888888888888888888888888888888\n"
                "8888888888888888888888888888888888888888888888888888888888888888888888888888\n"
                "8888888888888888888888888888888888888888888888888888888888888888888888888888\n"
                "8888888888888888888888888888888888888888888888888888888888888888888888888888\n"
                "8888888888888888888888888888888888888888888888888888888888888888888888888888\n"
                "8888888888888888888888888888888888888888888888888888888888888888888888888888\n"
                "8888888888888888888888888888888888888888888888888888888888888888888888888888\n"
                "8888888888888888888888888888888888888888888888888888888888888888888888888888\n"
                "8888888888888888888888888888888888888888888888888888888888888888888888888888\n"
                "8888888888888888888888888888888888888888888888888888888888888888888888888888\n"
            },
            {
                8, 8,
                0x88,
                "8888888888888888888888888888888888888888888888888888888888888888888888888888\n"
                "8888888888888888888888888888888888888888888888888888888888888888888888888888\n"
                "8888888888888888888888888888888888888888888888888888888888888888888888888888\n"
                "8888888888888888888888888888888888888888888888888888888888888888888888888888\n"
                "8888888888888888888888888888888888888888888888888888888888888888888888888888\n"
                "8888888888888888888888888888888888888888888888888888888888888888888888888888\n"
                "8888888888888888888888888888888888888888888888888888888888888888888888888888\n"
                "8888888888888888888888888888888888888888888888888888888888888888888888888888\n"
                "8888888888888888888888888888888888888888888888888888888888888888888888888888\n"
                "8888888888888888888888888888888888888888888888888888888888888888888888888888\n"
                "8888888888888888888888888888888888888888888888888888888888888888888888888888\n"
                "8888888888888888888888888888888888888888888888888888888888888888888888888888\n"
                "8888888888888888888888888888888888888888888888888888888888888888888888888888\n"
                "8888888888888888888888888888888888888888888888888888888888888888888888888888\n"
                "8888888888888888888888888888888888888888888888888888888888888888888888888888\n"
                "8888888888888888888888888888888888888888888888888888888888888888888888888888\n"
                "8888888888888888888888888888888888888888888888888888888888888888888888888888\n"
                "8888888888888888888800008888888888888888888888880000000000008888888800000000\n"
                "8888888888888888888800008888888888888888888800000000000000000088888800000000\n"
                "8888888888888888888800008888888888888888888800008888888888880088888800008888\n"
                "8888888888888888888800008888888888888888880000008888888888888888888800008888\n"
                "8888888888888888888800008888888888888888880000008888888888888888888800008888\n"
                "8888888888888888888800008888888888888888880000008888888888888888888800008888\n"
                "8888888888888888888800008888888888888888888800000088888888888888888800008888\n"
                "8888888888888888888800008888888888888888888888000000008888888888888800008888\n"
                "8888888888888888888800008888888888888888888888880000000000888888888800000000\n"
                "8888888888888888888800008888888888888888888888888888000000008888888800000000\n"
                "8888888888888888888800008888888888888888888888888888888800000088888800008888\n"
                "8888888888888888888800008888888888888888888888888888888888000088888800008888\n"
                "8888888888888888888800008888888888888888888888888888888888000088888800008888\n"
                "8888888888888888888800008888888888888888888888888888888888000088888800008888\n"
                "8888888888888888888800008888888888888888880088888888888800000088888800008888\n"
                "8888888888888888888800000000000000000088880000000000000000008888888800008888\n"
            },
            {
                -8, 8,
                0x88,
                "8888888888888888888888888888888888888888888888888888888888888888888888888888\n"
                "8888888888888888888888888888888888888888888888888888888888888888888888888888\n"
                "8888888888888888888888888888888888888888888888888888888888888888888888888888\n"
                "8888888888888888888888888888888888888888888888888888888888888888888888888888\n"
                "8888888888888888888888888888888888888888888888888888888888888888888888888888\n"
                "8888888888888888888888888888888888888888888888888888888888888888888888888888\n"
                "8888888888888888888888888888888888888888888888888888888888888888888888888888\n"
                "8888888888888888888888888888888888888888888888888888888888888888888888888888\n"
                "8888888888888888888888888888888888888888888888888888888888888888888888888888\n"
                "8888888888888888888888888888888888888888888888888888888888888888888888888888\n"
                "8888888888888888888888888888888888888888888888888888888888888888888888888888\n"
                "8888888888888888888888888888888888888888888888888888888888888888888888888888\n"
                "8888888888888888888888888888888888888888888888888888888888888888888888888888\n"
                "8888888888888888888888888888888888888888888888888888888888888888888888888888\n"
                "8888888888888888888888888888888888888888888888888888888888888888888888888888\n"
                "8888888888888888888888888888888888888888888888888888888888888888888888888888\n"
                "8888888888888888888888888888888888888888888888888888888888888888888888888888\n"
                "8888888888888888000000000000888888880000000000000000888888888888888888888888\n"
                "8888888888880000000000000000008888880000000000000000008888888888888888888888\n"
                "8888888888880000888888888888008888880000888888888800000088888888888888888888\n"
                "8888888888000000888888888888888888880000888888888888000088888888888888888888\n"
                "8888888888000000888888888888888888880000888888888888000000888888888888888888\n"
                "8888888888000000888888888888888888880000888888888888000000888888888888888888\n"
                "8888888888880000008888888888888888880000888888888888000088888888888888888888\n"
                "8888888888888800000000888888888888880000888888888800000088888888888888888888\n"
                "8888888888888888000000000088888888880000000000000000008888888888888888888888\n"
                "8888888888888888888800000000888888880000000000000088888888888888888888888888\n"
                "8888888888888888888888880000008888880000888888888888888888888888888888888888\n"
                "8888888888888888888888888800008888880000888888888888888888888888888888888888\n"
                "8888888888888888888888888800008888880000888888888888888888888888888888888888\n"
                "8888888888888888888888888800008888880000888888888888888888888888888888888888\n"
                "8888888888008888888888880000008888880000888888888888888888888888888888888888\n"
                "0000008888000000000000000000888888880000888888888888888888888888888888888888\n"
            },
            {
                0, 0,
                0,
                NULL
            }
        };

        static const bitmap_pattern_t max_patterns[] =
        {
            {
                0, 0,
                0x88,
                "8888888888888888888888888888888888888888888888888888888888888888888888888888\n"
                "8888888888888888888888888888888888888888888888888888888888888888888888888888\n"
                "8888888888888888888888888888888888888888888888888888888888888888888888888888\n"
                "8888888888888888888888888888888888888888888888888888888888888888888888888888\n"
                "8888888888888888888888888888888888888888888888888888888888888888888888888888\n"
                "8888888888888888888888888888888888888888888888888888888888888888888888888888\n"
                "8888888888888888888888888888888888888888888888888888888888888888888888888888\n"
                "8888888888888888888888888888888888888888888888888888888888888888888888888888\n"
                "8888888888888888888888888888888888888888888888888888888888888888888888888888\n"
                "8888ffff888888888888888888888888ffffffffffff88888888ffffffffffffffff88888888\n"
                "8888ffff88888888888888888888ffffffffffffffffff888888ffffffffffffffffff888888\n"
                "8888ffff88888888888888888888ffff888888888888ff888888ffff8888888888ffffff8888\n"
                "8888ffff888888888888888888ffffff88888888888888888888ffff888888888888ffff8888\n"
                "8888ffff888888888888888888ffffff88888888888888888888ffff888888888888ffffff88\n"
                "8888ffff888888888888888888ffffff88888888888888888888ffff888888888888ffffff88\n"
                "8888ffff88888888888888888888ffffff888888888888888888ffff888888888888ffff8888\n"
                "8888ffff8888888888888888888888ffffffff88888888888888ffff8888888888ffffff8888\n"
                "8888ffff888888888888888888888888ffffffffff8888888888ffffffffffffffffff888888\n"
                "8888ffff8888888888888888888888888888ffffffff88888888ffffffffffffff8888888888\n"
                "8888ffff88888888888888888888888888888888ffffff888888ffff88888888888888888888\n"
                "8888ffff8888888888888888888888888888888888ffff888888ffff88888888888888888888\n"
                "8888ffff8888888888888888888888888888888888ffff888888ffff88888888888888888888\n"
                "8888ffff8888888888888888888888888888888888ffff888888ffff88888888888888888888\n"
                "8888ffff888888888888888888ff888888888888ffffff888888ffff88888888888888888888\n"
                "8888ffffffffffffffffff8888ffffffffffffffffff88888888ffff88888888888888888888\n"
                "8888ffffffffffffffffff888888ffffffffffff888888888888ffff88888888888888888888\n"
                "8888888888888888888888888888888888888888888888888888888888888888888888888888\n"
                "8888888888888888888888888888888888888888888888888888888888888888888888888888\n"
                "8888888888888888888888888888888888888888888888888888888888888888888888888888\n"
                "8888888888888888888888888888888888888888888888888888888888888888888888888888\n"
                "8888888888888888888888888888888888888888888888888888888888888888888888888888\n"
                "8888888888888888888888888888888888888888888888888888888888888888888888888888\n"
                "8888888888888888888888888888888888888888888888888888888888888888888888888888\n"
            },
            {
                -8, -8,
                0x88,
                "8888888888888888888888888888888888888888888888888888888888888888888888888888\n"
                "8888888888888888ffffffffffff88888888ffffffffffffffff888888888888888888888888\n"
                "888888888888ffffffffffffffffff888888ffffffffffffffffff8888888888888888888888\n"
                "888888888888ffff888888888888ff888888ffff8888888888ffffff88888888888888888888\n"
                "8888888888ffffff88888888888888888888ffff888888888888ffff88888888888888888888\n"
                "8888888888ffffff88888888888888888888ffff888888888888ffffff888888888888888888\n"
                "8888888888ffffff88888888888888888888ffff888888888888ffffff888888888888888888\n"
                "888888888888ffffff888888888888888888ffff888888888888ffff88888888888888888888\n"
                "88888888888888ffffffff88888888888888ffff8888888888ffffff88888888888888888888\n"
                "8888888888888888ffffffffff8888888888ffffffffffffffffff8888888888888888888888\n"
                "88888888888888888888ffffffff88888888ffffffffffffff88888888888888888888888888\n"
                "888888888888888888888888ffffff888888ffff888888888888888888888888888888888888\n"
                "88888888888888888888888888ffff888888ffff888888888888888888888888888888888888\n"
                "88888888888888888888888888ffff888888ffff888888888888888888888888888888888888\n"
                "88888888888888888888888888ffff888888ffff888888888888888888888888888888888888\n"
                "8888888888ff888888888888ffffff888888ffff888888888888888888888888888888888888\n"
                "ffffff8888ffffffffffffffffff88888888ffff888888888888888888888888888888888888\n"
                "ffffff888888ffffffffffff888888888888ffff888888888888888888888888888888888888\n"
                "8888888888888888888888888888888888888888888888888888888888888888888888888888\n"
                "8888888888888888888888888888888888888888888888888888888888888888888888888888\n"
                "8888888888888888888888888888888888888888888888888888888888888888888888888888\n"
                "8888888888888888888888888888888888888888888888888888888888888888888888888888\n"
                "8888888888888888888888888888888888888888888888888888888888888888888888888888\n"
                "8888888888888888888888888888888888888888888888888888888888888888888888888888\n"
                "8888888888888888888888888888888888888888888888888888888888888888888888888888\n"
                "8888888888888888888888888888888888888888888888888888888888888888888888888888\n"
                "8888888888888888888888888888888888888888888888888888888888888888888888888888\n"
                "8888888888888888888888888888888888888888888888888888888888888888888888888888\n"
                "8888888888888888888888888888888888888888888888888888888888888888888888888888\n"
                "8888888888888888888888888888888888888888888888888888888888888888888888888888\n"
                "8888888888888888888888888888888888888888888888888888888888888888888888888888\n"
                "8888888888888888888888888888888888888888888888888888888888888888888888888888\n"
                "8888888888888888888888888888888888888888888888888888888888888888888888888888\n"
            },
            {
                8, -8,
                0x88,
                "8888888888888888888888888888888888888888888888888888888888888888888888888888\n"
                "88888888888888888888ffff888888888888888888888888ffffffffffff88888888ffffffff\n"
                "88888888888888888888ffff88888888888888888888ffffffffffffffffff888888ffffffff\n"
                "88888888888888888888ffff88888888888888888888ffff888888888888ff888888ffff8888\n"
                "88888888888888888888ffff888888888888888888ffffff88888888888888888888ffff8888\n"
                "88888888888888888888ffff888888888888888888ffffff88888888888888888888ffff8888\n"
                "88888888888888888888ffff888888888888888888ffffff88888888888888888888ffff8888\n"
                "88888888888888888888ffff88888888888888888888ffffff888888888888888888ffff8888\n"
                "88888888888888888888ffff8888888888888888888888ffffffff88888888888888ffff8888\n"
                "88888888888888888888ffff888888888888888888888888ffffffffff8888888888ffffffff\n"
                "88888888888888888888ffff8888888888888888888888888888ffffffff88888888ffffffff\n"
                "88888888888888888888ffff88888888888888888888888888888888ffffff888888ffff8888\n"
                "88888888888888888888ffff8888888888888888888888888888888888ffff888888ffff8888\n"
                "88888888888888888888ffff8888888888888888888888888888888888ffff888888ffff8888\n"
                "88888888888888888888ffff8888888888888888888888888888888888ffff888888ffff8888\n"
                "88888888888888888888ffff888888888888888888ff888888888888ffffff888888ffff8888\n"
                "88888888888888888888ffffffffffffffffff8888ffffffffffffffffff88888888ffff8888\n"
                "88888888888888888888ffffffffffffffffff888888ffffffffffff888888888888ffff8888\n"
                "8888888888888888888888888888888888888888888888888888888888888888888888888888\n"
                "8888888888888888888888888888888888888888888888888888888888888888888888888888\n"
                "8888888888888888888888888888888888888888888888888888888888888888888888888888\n"
                "8888888888888888888888888888888888888888888888888888888888888888888888888888\n"
                "8888888888888888888888888888888888888888888888888888888888888888888888888888\n"
                "8888888888888888888888888888888888888888888888888888888888888888888888888888\n"
                "8888888888888888888888888888888888888888888888888888888888888888888888888888\n"
                "8888888888888888888888888888888888888888888888888888888888888888888888888888\n"
                "8888888888888888888888888888888888888888888888888888888888888888888888888888\n"
                "8888888888888888888888888888888888888888888888888888888888888888888888888888\n"
                "8888888888888888888888888888888888888888888888888888888888888888888888888888\n"
                "8888888888888888888888888888888888888888888888888888888888888888888888888888\n"
                "8888888888888888888888888888888888888888888888888888888888888888888888888888\n"
                "8888888888888888888888888888888888888888888888888888888888888888888888888888\n"
                "8888888888888888888888888888888888888888888888888888888888888888888888888888\n"
            },
            {
                8, 8,
                0x88,
                "8888888888888888888888888888888888888888888888888888888888888888888888888888\n"
                "8888888888888888888888888888888888888888888888888888888888888888888888888888\n"
                "8888888888888888888888888888888888888888888888888888888888888888888888888888\n"
                "8888888888888888888888888888888888888888888888888888888888888888888888888888\n"
                "8888888888888888888888888888888888888888888888888888888888888888888888888888\n"
                "8888888888888888888888888888888888888888888888888888888888888888888888888888\n"
                "8888888888888888888888888888888888888888888888888888888888888888888888888888\n"
                "8888888888888888888888888888888888888888888888888888888888888888888888888888\n"
                "8888888888888888888888888888888888888888888888888888888888888888888888888888\n"
                "8888888888888888888888888888888888888888888888888888888888888888888888888888\n"
                "8888888888888888888888888888888888888888888888888888888888888888888888888888\n"
                "8888888888888888888888888888888888888888888888888888888888888888888888888888\n"
                "8888888888888888888888888888888888888888888888888888888888888888888888888888\n"
                "8888888888888888888888888888888888888888888888888888888888888888888888888888\n"
                "8888888888888888888888888888888888888888888888888888888888888888888888888888\n"
                "8888888888888888888888888888888888888888888888888888888888888888888888888888\n"
                "8888888888888888888888888888888888888888888888888888888888888888888888888888\n"
                "88888888888888888888ffff888888888888888888888888ffffffffffff88888888ffffffff\n"
                "88888888888888888888ffff88888888888888888888ffffffffffffffffff888888ffffffff\n"
                "88888888888888888888ffff88888888888888888888ffff888888888888ff888888ffff8888\n"
                "88888888888888888888ffff888888888888888888ffffff88888888888888888888ffff8888\n"
                "88888888888888888888ffff888888888888888888ffffff88888888888888888888ffff8888\n"
                "88888888888888888888ffff888888888888888888ffffff88888888888888888888ffff8888\n"
                "88888888888888888888ffff88888888888888888888ffffff888888888888888888ffff8888\n"
                "88888888888888888888ffff8888888888888888888888ffffffff88888888888888ffff8888\n"
                "88888888888888888888ffff888888888888888888888888ffffffffff8888888888ffffffff\n"
                "88888888888888888888ffff8888888888888888888888888888ffffffff88888888ffffffff\n"
                "88888888888888888888ffff88888888888888888888888888888888ffffff888888ffff8888\n"
                "88888888888888888888ffff8888888888888888888888888888888888ffff888888ffff8888\n"
                "88888888888888888888ffff8888888888888888888888888888888888ffff888888ffff8888\n"
                "88888888888888888888ffff8888888888888888888888888888888888ffff888888ffff8888\n"
                "88888888888888888888ffff888888888888888888ff888888888888ffffff888888ffff8888\n"
                "88888888888888888888ffffffffffffffffff8888ffffffffffffffffff88888888ffff8888\n"
            },
            {
                -8, 8,
                0x88,
                "8888888888888888888888888888888888888888888888888888888888888888888888888888\n"
                "8888888888888888888888888888888888888888888888888888888888888888888888888888\n"
                "8888888888888888888888888888888888888888888888888888888888888888888888888888\n"
                "8888888888888888888888888888888888888888888888888888888888888888888888888888\n"
                "8888888888888888888888888888888888888888888888888888888888888888888888888888\n"
                "8888888888888888888888888888888888888888888888888888888888888888888888888888\n"
                "8888888888888888888888888888888888888888888888888888888888888888888888888888\n"
                "8888888888888888888888888888888888888888888888888888888888888888888888888888\n"
                "8888888888888888888888888888888888888888888888888888888888888888888888888888\n"
                "8888888888888888888888888888888888888888888888888888888888888888888888888888\n"
                "8888888888888888888888888888888888888888888888888888888888888888888888888888\n"
                "8888888888888888888888888888888888888888888888888888888888888888888888888888\n"
                "8888888888888888888888888888888888888888888888888888888888888888888888888888\n"
                "8888888888888888888888888888888888888888888888888888888888888888888888888888\n"
                "8888888888888888888888888888888888888888888888888888888888888888888888888888\n"
                "8888888888888888888888888888888888888888888888888888888888888888888888888888\n"
                "8888888888888888888888888888888888888888888888888888888888888888888888888888\n"
                "8888888888888888ffffffffffff88888888ffffffffffffffff888888888888888888888888\n"
                "888888888888ffffffffffffffffff888888ffffffffffffffffff8888888888888888888888\n"
                "888888888888ffff888888888888ff888888ffff8888888888ffffff88888888888888888888\n"
                "8888888888ffffff88888888888888888888ffff888888888888ffff88888888888888888888\n"
                "8888888888ffffff88888888888888888888ffff888888888888ffffff888888888888888888\n"
                "8888888888ffffff88888888888888888888ffff888888888888ffffff888888888888888888\n"
                "888888888888ffffff888888888888888888ffff888888888888ffff88888888888888888888\n"
                "88888888888888ffffffff88888888888888ffff8888888888ffffff88888888888888888888\n"
                "8888888888888888ffffffffff8888888888ffffffffffffffffff8888888888888888888888\n"
                "88888888888888888888ffffffff88888888ffffffffffffff88888888888888888888888888\n"
                "888888888888888888888888ffffff888888ffff888888888888888888888888888888888888\n"
                "88888888888888888888888888ffff888888ffff888888888888888888888888888888888888\n"
                "88888888888888888888888888ffff888888ffff888888888888888888888888888888888888\n"
                "88888888888888888888888888ffff888888ffff888888888888888888888888888888888888\n"
                "8888888888ff888888888888ffffff888888ffff888888888888888888888888888888888888\n"
                "ffffff8888ffffffffffffffffff88888888ffff888888888888888888888888888888888888\n"
            },
            {
                0, 0,
                0,
                NULL
            }
        };

        static const bitmap_pattern_t min_patterns[] =
        {
            {
                0, 0,
                0x88,
                "0000000000000000000000000000000000000000000000000000000000000000000000000000\n"
                "0000000000000000000000000000000000000000000000000000000000000000000000000000\n"
                "0000000000000000000000000000000000000000000000000000000000000000000000000000\n"
                "0000000000000000000000000000000000000000000000000000000000000000000000000000\n"
                "0000000000000000000000000000000000000000000000000000000000000000000000000000\n"
                "0000000000000000000000000000000000000000000000000000000000000000000000000000\n"
                "0000000000000000000000000000000000000000000000000000000000000000000000000000\n"
                "0000000000000000000000000000000000000000000000000000000000000000000000000000\n"
                "0000000000000000000000000000000000000000000000000000000000000000000000000000\n"
                "0000888800000000000000000000000088888888888800000000888888888888888800000000\n"
                "0000888800000000000000000000888888888888888888000000888888888888888888000000\n"
                "0000888800000000000000000000888800000000000088000000888800000000008888880000\n"
                "0000888800000000000000000088888800000000000000000000888800000000000088880000\n"
                "0000888800000000000000000088888800000000000000000000888800000000000088888800\n"
                "0000888800000000000000000088888800000000000000000000888800000000000088888800\n"
                "0000888800000000000000000000888888000000000000000000888800000000000088880000\n"
                "0000888800000000000000000000008888888800000000000000888800000000008888880000\n"
                "0000888800000000000000000000000088888888880000000000888888888888888888000000\n"
                "0000888800000000000000000000000000008888888800000000888888888888880000000000\n"
                "0000888800000000000000000000000000000000888888000000888800000000000000000000\n"
                "0000888800000000000000000000000000000000008888000000888800000000000000000000\n"
                "0000888800000000000000000000000000000000008888000000888800000000000000000000\n"
                "0000888800000000000000000000000000000000008888000000888800000000000000000000\n"
                "0000888800000000000000000088000000000000888888000000888800000000000000000000\n"
                "0000888888888888888888000088888888888888888800000000888800000000000000000000\n"
                "0000888888888888888888000000888888888888000000000000888800000000000000000000\n"
                "0000000000000000000000000000000000000000000000000000000000000000000000000000\n"
                "0000000000000000000000000000000000000000000000000000000000000000000000000000\n"
                "0000000000000000000000000000000000000000000000000000000000000000000000000000\n"
                "0000000000000000000000000000000000000000000000000000000000000000000000000000\n"
                "0000000000000000000000000000000000000000000000000000000000000000000000000000\n"
                "0000000000000000000000000000000000000000000000000000000000000000000000000000\n"
                "0000000000000000000000000000000000000000000000000000000000000000000000000000\n"
            },
            {
                -8, -8,
                0x88,
                "0000000000000000000000000000000000000000000000000000000000008888888888888888\n"
                "0000000000000000888888888888000000008888888888888888000000008888888888888888\n"
                "0000000000008888888888888888880000008888888888888888880000008888888888888888\n"
                "0000000000008888000000000000880000008888000000000088888800008888888888888888\n"
                "0000000000888888000000000000000000008888000000000000888800008888888888888888\n"
                "0000000000888888000000000000000000008888000000000000888888008888888888888888\n"
                "0000000000888888000000000000000000008888000000000000888888008888888888888888\n"
                "0000000000008888880000000000000000008888000000000000888800008888888888888888\n"
                "0000000000000088888888000000000000008888000000000088888800008888888888888888\n"
                "0000000000000000888888888800000000008888888888888888880000008888888888888888\n"
                "0000000000000000000088888888000000008888888888888800000000008888888888888888\n"
                "0000000000000000000000008888880000008888000000000000000000008888888888888888\n"
                "0000000000000000000000000088880000008888000000000000000000008888888888888888\n"
                "0000000000000000000000000088880000008888000000000000000000008888888888888888\n"
                "0000000000000000000000000088880000008888000000000000000000008888888888888888\n"
                "0000000000880000000000008888880000008888000000000000000000008888888888888888\n"
                "8888880000888888888888888888000000008888000000000000000000008888888888888888\n"
                "8888880000008888888888880000000000008888000000000000000000008888888888888888\n"
                "0000000000000000000000000000000000000000000000000000000000008888888888888888\n"
                "0000000000000000000000000000000000000000000000000000000000008888888888888888\n"
                "0000000000000000000000000000000000000000000000000000000000008888888888888888\n"
                "0000000000000000000000000000000000000000000000000000000000008888888888888888\n"
                "0000000000000000000000000000000000000000000000000000000000008888888888888888\n"
                "0000000000000000000000000000000000000000000000000000000000008888888888888888\n"
                "0000000000000000000000000000000000000000000000000000000000008888888888888888\n"
                "8888888888888888888888888888888888888888888888888888888888888888888888888888\n"
                "8888888888888888888888888888888888888888888888888888888888888888888888888888\n"
                "8888888888888888888888888888888888888888888888888888888888888888888888888888\n"
                "8888888888888888888888888888888888888888888888888888888888888888888888888888\n"
                "8888888888888888888888888888888888888888888888888888888888888888888888888888\n"
                "8888888888888888888888888888888888888888888888888888888888888888888888888888\n"
                "8888888888888888888888888888888888888888888888888888888888888888888888888888\n"
                "8888888888888888888888888888888888888888888888888888888888888888888888888888\n"
            },
            {
                8, -8,
                0x88,
                "8888888888888888000000000000000000000000000000000000000000000000000000000000\n"
                "8888888888888888000088880000000000000000000000008888888888880000000088888888\n"
                "8888888888888888000088880000000000000000000088888888888888888800000088888888\n"
                "8888888888888888000088880000000000000000000088880000000000008800000088880000\n"
                "8888888888888888000088880000000000000000008888880000000000000000000088880000\n"
                "8888888888888888000088880000000000000000008888880000000000000000000088880000\n"
                "8888888888888888000088880000000000000000008888880000000000000000000088880000\n"
                "8888888888888888000088880000000000000000000088888800000000000000000088880000\n"
                "8888888888888888000088880000000000000000000000888888880000000000000088880000\n"
                "8888888888888888000088880000000000000000000000008888888888000000000088888888\n"
                "8888888888888888000088880000000000000000000000000000888888880000000088888888\n"
                "8888888888888888000088880000000000000000000000000000000088888800000088880000\n"
                "8888888888888888000088880000000000000000000000000000000000888800000088880000\n"
                "8888888888888888000088880000000000000000000000000000000000888800000088880000\n"
                "8888888888888888000088880000000000000000000000000000000000888800000088880000\n"
                "8888888888888888000088880000000000000000008800000000000088888800000088880000\n"
                "8888888888888888000088888888888888888800008888888888888888880000000088880000\n"
                "8888888888888888000088888888888888888800000088888888888800000000000088880000\n"
                "8888888888888888000000000000000000000000000000000000000000000000000000000000\n"
                "8888888888888888000000000000000000000000000000000000000000000000000000000000\n"
                "8888888888888888000000000000000000000000000000000000000000000000000000000000\n"
                "8888888888888888000000000000000000000000000000000000000000000000000000000000\n"
                "8888888888888888000000000000000000000000000000000000000000000000000000000000\n"
                "8888888888888888000000000000000000000000000000000000000000000000000000000000\n"
                "8888888888888888000000000000000000000000000000000000000000000000000000000000\n"
                "8888888888888888888888888888888888888888888888888888888888888888888888888888\n"
                "8888888888888888888888888888888888888888888888888888888888888888888888888888\n"
                "8888888888888888888888888888888888888888888888888888888888888888888888888888\n"
                "8888888888888888888888888888888888888888888888888888888888888888888888888888\n"
                "8888888888888888888888888888888888888888888888888888888888888888888888888888\n"
                "8888888888888888888888888888888888888888888888888888888888888888888888888888\n"
                "8888888888888888888888888888888888888888888888888888888888888888888888888888\n"
                "8888888888888888888888888888888888888888888888888888888888888888888888888888\n"
            },
            {
                8, 8,
                0x88,
                "8888888888888888888888888888888888888888888888888888888888888888888888888888\n"
                "8888888888888888888888888888888888888888888888888888888888888888888888888888\n"
                "8888888888888888888888888888888888888888888888888888888888888888888888888888\n"
                "8888888888888888888888888888888888888888888888888888888888888888888888888888\n"
                "8888888888888888888888888888888888888888888888888888888888888888888888888888\n"
                "8888888888888888888888888888888888888888888888888888888888888888888888888888\n"
                "8888888888888888888888888888888888888888888888888888888888888888888888888888\n"
                "8888888888888888888888888888888888888888888888888888888888888888888888888888\n"
                "8888888888888888000000000000000000000000000000000000000000000000000000000000\n"
                "8888888888888888000000000000000000000000000000000000000000000000000000000000\n"
                "8888888888888888000000000000000000000000000000000000000000000000000000000000\n"
                "8888888888888888000000000000000000000000000000000000000000000000000000000000\n"
                "8888888888888888000000000000000000000000000000000000000000000000000000000000\n"
                "8888888888888888000000000000000000000000000000000000000000000000000000000000\n"
                "8888888888888888000000000000000000000000000000000000000000000000000000000000\n"
                "8888888888888888000000000000000000000000000000000000000000000000000000000000\n"
                "8888888888888888000000000000000000000000000000000000000000000000000000000000\n"
                "8888888888888888000088880000000000000000000000008888888888880000000088888888\n"
                "8888888888888888000088880000000000000000000088888888888888888800000088888888\n"
                "8888888888888888000088880000000000000000000088880000000000008800000088880000\n"
                "8888888888888888000088880000000000000000008888880000000000000000000088880000\n"
                "8888888888888888000088880000000000000000008888880000000000000000000088880000\n"
                "8888888888888888000088880000000000000000008888880000000000000000000088880000\n"
                "8888888888888888000088880000000000000000000088888800000000000000000088880000\n"
                "8888888888888888000088880000000000000000000000888888880000000000000088880000\n"
                "8888888888888888000088880000000000000000000000008888888888000000000088888888\n"
                "8888888888888888000088880000000000000000000000000000888888880000000088888888\n"
                "8888888888888888000088880000000000000000000000000000000088888800000088880000\n"
                "8888888888888888000088880000000000000000000000000000000000888800000088880000\n"
                "8888888888888888000088880000000000000000000000000000000000888800000088880000\n"
                "8888888888888888000088880000000000000000000000000000000000888800000088880000\n"
                "8888888888888888000088880000000000000000008800000000000088888800000088880000\n"
                "8888888888888888000088888888888888888800008888888888888888880000000088880000\n"
            },
            {
                -8, 8,
                0x88,
                "8888888888888888888888888888888888888888888888888888888888888888888888888888\n"
                "8888888888888888888888888888888888888888888888888888888888888888888888888888\n"
                "8888888888888888888888888888888888888888888888888888888888888888888888888888\n"
                "8888888888888888888888888888888888888888888888888888888888888888888888888888\n"
                "8888888888888888888888888888888888888888888888888888888888888888888888888888\n"
                "8888888888888888888888888888888888888888888888888888888888888888888888888888\n"
                "8888888888888888888888888888888888888888888888888888888888888888888888888888\n"
                "8888888888888888888888888888888888888888888888888888888888888888888888888888\n"
                "0000000000000000000000000000000000000000000000000000000000008888888888888888\n"
                "0000000000000000000000000000000000000000000000000000000000008888888888888888\n"
                "0000000000000000000000000000000000000000000000000000000000008888888888888888\n"
                "0000000000000000000000000000000000000000000000000000000000008888888888888888\n"
                "0000000000000000000000000000000000000000000000000000000000008888888888888888\n"
                "0000000000000000000000000000000000000000000000000000000000008888888888888888\n"
                "0000000000000000000000000000000000000000000000000000000000008888888888888888\n"
                "0000000000000000000000000000000000000000000000000000000000008888888888888888\n"
                "0000000000000000000000000000000000000000000000000000000000008888888888888888\n"
                "0000000000000000888888888888000000008888888888888888000000008888888888888888\n"
                "0000000000008888888888888888880000008888888888888888880000008888888888888888\n"
                "0000000000008888000000000000880000008888000000000088888800008888888888888888\n"
                "0000000000888888000000000000000000008888000000000000888800008888888888888888\n"
                "0000000000888888000000000000000000008888000000000000888888008888888888888888\n"
                "0000000000888888000000000000000000008888000000000000888888008888888888888888\n"
                "0000000000008888880000000000000000008888000000000000888800008888888888888888\n"
                "0000000000000088888888000000000000008888000000000088888800008888888888888888\n"
                "0000000000000000888888888800000000008888888888888888880000008888888888888888\n"
                "0000000000000000000088888888000000008888888888888800000000008888888888888888\n"
                "0000000000000000000000008888880000008888000000000000000000008888888888888888\n"
                "0000000000000000000000000088880000008888000000000000000000008888888888888888\n"
                "0000000000000000000000000088880000008888000000000000000000008888888888888888\n"
                "0000000000000000000000000088880000008888000000000000000000008888888888888888\n"
                "0000000000880000000000008888880000008888000000000000000000008888888888888888\n"
                "8888880000888888888888888888000000008888000000000000000000008888888888888888\n"
            },
            {
                0, 0,
                0,
                NULL
            }
        };

    } /* namespace test */
} /* namespace lsp */

UTEST_BEGIN("dsp.bitmap", b1b8)

    void test_pattern(const char *name, bitmap_func_b1b8_t func, const bitmap_pattern_t *pattern)
    {
        printf("Testing '%s' at point (%d, %d)\n", name, int(pattern->x), int(pattern->y));

        dsp::bitmap_t tmp;
        ByteBuffer bbuf(bitmap.width * bitmap.height);

        tmp.width   = bitmap.width;
        tmp.height  = bitmap.height;
        tmp.stride  = tmp.width;
        tmp.data    = bbuf.data();
        memset(bbuf.data(), pattern->fill, bitmap.width * bitmap.height);

        func(&tmp, &bitmap, pattern->x, pattern->y);

        UTEST_ASSERT(!bbuf.corrupted());

        char *result = rasterize(&tmp);
        UTEST_ASSERT(result != NULL);
        lsp_finally { free(result); };

        if (strcmp(result, pattern->data) != 0)
        {
            UTEST_FAIL_MSG("Faled rasterization:\ngot pattern:\n%s\nexpected pattern:\n%s\n", result, pattern->data);
        }
    }

    void test_patterns(const char *name, bitmap_func_b1b8_t func, const bitmap_pattern_t * list)
    {
        if (!UTEST_SUPPORTED(func))
            return;

        for (; list->data != NULL; ++list)
            test_pattern(name, func, list);
    }


    UTEST_MAIN
    {
        using namespace lsp;

        #define TEST_PATTERNS(func, list) \
            test_patterns(#func, func, list)

        TEST_PATTERNS(generic::bitmap_put_b1b8, put_patterns);
        TEST_PATTERNS(generic::bitmap_add_b1b8, add_patterns);
        TEST_PATTERNS(generic::bitmap_sub_b1b8, sub_patterns);
        TEST_PATTERNS(generic::bitmap_max_b1b8, max_patterns);
        TEST_PATTERNS(generic::bitmap_min_b1b8, min_patterns);
    }

UTEST_END;


